@extends('layouts.admin')

@section('title', 'Branch Management | IMMIA FINANCE LIMITED')

@section('page-title', 'Branch Management')
@section('page-description', 'Manage your branches')

@section('content')
<!-- Header with Add Button -->
<div class="flex justify-between items-center mb-6">
    <h2 class="text-xl font-semibold text-gray-800">Branches</h2>
    <button onclick="openCreateModal()" class="bg-blue-600 hover:bg-blue-700 text-white px-4 py-2 rounded-lg font-semibold flex items-center space-x-2 transition-colors">
        <i class="fas fa-plus"></i>
        <span>New Branch</span>
    </button>
</div>

<!-- Branches Table -->
<div class="bg-white rounded-xl shadow-sm border overflow-hidden">
    <div class="overflow-x-auto">
        <table class="min-w-full divide-y divide-gray-200">
            <thead class="bg-gray-50">
                <tr>
                    <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Branch Code</th>
                    <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Branch Name</th>
                    <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Setup Date</th>
                    <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Status</th>
                    <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Created By</th>
                    <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Actions</th>
                </tr>
            </thead>
            <tbody class="bg-white divide-y divide-gray-200">
                @forelse($branches as $branch)
                <tr class="hover:bg-gray-50 transition-colors">
                    <td class="px-6 py-4 whitespace-nowrap">
                        <div class="text-sm font-medium text-gray-900">{{ $branch->branchcode }}</div>
                    </td>
                    <td class="px-6 py-4 whitespace-nowrap">
                        <div class="text-sm text-gray-900">{{ $branch->branchname }}</div>
                    </td>
                    <td class="px-6 py-4 whitespace-nowrap">
                        <div class="text-sm text-gray-900">{{ $branch->setupdate->format('M d, Y') }}</div>
                    </td>
                    <td class="px-6 py-4 whitespace-nowrap">
                        <span class="inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium {{ $branch->status ? 'bg-green-100 text-green-800' : 'bg-red-100 text-red-800' }}">
                            <i class="fas fa-circle mr-1" style="font-size: 6px;"></i>
                            {{ $branch->status_text }}
                        </span>
                    </td>
                    <td class="px-6 py-4 whitespace-nowrap">
                        <div class="text-sm text-gray-900">{{ $branch->user->name ?? 'N/A' }}</div>
                    </td>
                    <td class="px-6 py-4 whitespace-nowrap text-sm font-medium">
                        <div class="flex items-center space-x-3">
                            <button onclick="openEditModal({{ $branch->id }})" class="text-blue-600 hover:text-blue-900 transition-colors" title="Edit Branch">
                                <i class="fas fa-edit"></i>
                            </button>
                            <button onclick="toggleStatus({{ $branch->id }})" class="text-{{ $branch->status ? 'yellow' : 'green' }}-600 hover:text-{{ $branch->status ? 'yellow' : 'green' }}-900 transition-colors" title="{{ $branch->status ? 'Deactivate' : 'Activate' }} Branch">
                                <i class="fas fa-{{ $branch->status ? 'pause' : 'play' }}"></i>
                            </button>
                            <button onclick="confirmDelete({{ $branch->id }})" class="text-red-600 hover:text-red-900 transition-colors" title="Delete Branch">
                                <i class="fas fa-trash"></i>
                            </button>
                        </div>
                    </td>
                </tr>
                @empty
                <tr>
                    <td colspan="6" class="px-6 py-8 text-center">
                        <div class="flex flex-col items-center justify-center text-gray-500">
                            <i class="fas fa-code-branch text-4xl mb-4 text-gray-300"></i>
                            <p class="text-lg font-medium mb-2">No branches found</p>
                            <p class="text-sm mb-4">Get started by creating your first branch</p>
                            <button onclick="openCreateModal()" class="bg-blue-600 hover:bg-blue-700 text-white px-4 py-2 rounded-lg font-medium transition-colors">
                                <i class="fas fa-plus mr-2"></i>
                                Create First Branch
                            </button>
                        </div>
                    </td>
                </tr>
                @endforelse
            </tbody>
        </table>
    </div>
</div>

<!-- Create Branch Modal -->
<div id="createModal" class="fixed inset-0 bg-gray-600 bg-opacity-50 overflow-y-auto h-full w-full hidden z-50 transition-opacity">
    <div class="relative top-20 mx-auto p-5 border w-full max-w-md shadow-lg rounded-md bg-white transform transition-all">
        <div class="mt-3">
            <div class="flex justify-between items-center mb-4 pb-3 border-b">
                <h3 class="text-lg font-semibold text-gray-900">Create New Branch</h3>
                <button onclick="closeCreateModal()" class="text-gray-400 hover:text-gray-600 transition-colors">
                    <i class="fas fa-times text-lg"></i>
                </button>
            </div>
            
            <form id="createBranchForm">
                @csrf
                <div class="space-y-4">
                    <div>
                        <label for="branchcode" class="block text-sm font-medium text-gray-700 mb-1">Branch Code *</label>
                        <input type="text" name="branchcode" id="branchcode" required 
                               class="w-full border border-gray-300 rounded-lg shadow-sm py-2 px-3 focus:outline-none focus:ring-2 focus:ring-blue-500 focus:border-blue-500 transition-colors"
                               placeholder="e.g., SOL001">
                        <p class="text-xs text-gray-500 mt-1">Unique code for the branch</p>
                    </div>
                    
                    <div>
                        <label for="branchname" class="block text-sm font-medium text-gray-700 mb-1">Branch Name *</label>
                        <input type="text" name="branchname" id="branchname" required 
                               class="w-full border border-gray-300 rounded-lg shadow-sm py-2 px-3 focus:outline-none focus:ring-2 focus:ring-blue-500 focus:border-blue-500 transition-colors"
                               placeholder="e.g., Solwezi Main Branch">
                    </div>
                    
                    <div>
                        <label for="setupdate" class="block text-sm font-medium text-gray-700 mb-1">Setup Date *</label>
                        <input type="date" name="setupdate" id="setupdate" required 
                               class="w-full border border-gray-300 rounded-lg shadow-sm py-2 px-3 focus:outline-none focus:ring-2 focus:ring-blue-500 focus:border-blue-500 transition-colors">
                    </div>
                </div>
                
                <div class="flex justify-end space-x-3 mt-6 pt-4 border-t">
                    <button type="button" onclick="closeCreateModal()" class="bg-gray-300 hover:bg-gray-400 text-gray-800 py-2 px-4 rounded-lg font-medium transition-colors">
                        Cancel
                    </button>
                    <button type="submit" class="bg-blue-600 hover:bg-blue-700 text-white py-2 px-4 rounded-lg font-medium flex items-center space-x-2 transition-colors">
                        <i class="fas fa-save"></i>
                        <span>Create Branch</span>
                    </button>
                </div>
            </form>
        </div>
    </div>
</div>

<!-- Edit Branch Modal -->
<div id="editModal" class="fixed inset-0 bg-gray-600 bg-opacity-50 overflow-y-auto h-full w-full hidden z-50 transition-opacity">
    <div class="relative top-20 mx-auto p-5 border w-full max-w-md shadow-lg rounded-md bg-white transform transition-all">
        <div class="mt-3">
            <div class="flex justify-between items-center mb-4 pb-3 border-b">
                <h3 class="text-lg font-semibold text-gray-900">Edit Branch</h3>
                <button onclick="closeEditModal()" class="text-gray-400 hover:text-gray-600 transition-colors">
                    <i class="fas fa-times text-lg"></i>
                </button>
            </div>
            
            <form id="editBranchForm">
                @csrf
                @method('PUT')
                <input type="hidden" name="branch_id" id="edit_branch_id">
                
                <div class="space-y-4">
                    <div>
                        <label for="edit_branchcode" class="block text-sm font-medium text-gray-700 mb-1">Branch Code *</label>
                        <input type="text" name="branchcode" id="edit_branchcode" required 
                               class="w-full border border-gray-300 rounded-lg shadow-sm py-2 px-3 focus:outline-none focus:ring-2 focus:ring-blue-500 focus:border-blue-500 transition-colors"
                               placeholder="e.g., SOL001">
                    </div>
                    
                    <div>
                        <label for="edit_branchname" class="block text-sm font-medium text-gray-700 mb-1">Branch Name *</label>
                        <input type="text" name="branchname" id="edit_branchname" required 
                               class="w-full border border-gray-300 rounded-lg shadow-sm py-2 px-3 focus:outline-none focus:ring-2 focus:ring-blue-500 focus:border-blue-500 transition-colors"
                               placeholder="e.g., Solwezi Main Branch">
                    </div>
                    
                    <div>
                        <label for="edit_setupdate" class="block text-sm font-medium text-gray-700 mb-1">Setup Date *</label>
                        <input type="date" name="setupdate" id="edit_setupdate" required 
                               class="w-full border border-gray-300 rounded-lg shadow-sm py-2 px-3 focus:outline-none focus:ring-2 focus:ring-blue-500 focus:border-blue-500 transition-colors">
                    </div>
                </div>
                
                <div class="flex justify-end space-x-3 mt-6 pt-4 border-t">
                    <button type="button" onclick="closeEditModal()" class="bg-gray-300 hover:bg-gray-400 text-gray-800 py-2 px-4 rounded-lg font-medium transition-colors">
                        Cancel
                    </button>
                    <button type="submit" class="bg-blue-600 hover:bg-blue-700 text-white py-2 px-4 rounded-lg font-medium flex items-center space-x-2 transition-colors">
                        <i class="fas fa-save"></i>
                        <span>Update Branch</span>
                    </button>
                </div>
            </form>
        </div>
    </div>
</div>

<!-- Loading Overlay for Edit Modal -->
<div id="editLoading" class="fixed inset-0 bg-gray-600 bg-opacity-50 overflow-y-auto h-full w-full hidden z-50">
    <div class="relative top-20 mx-auto p-5 border w-full max-w-md shadow-lg rounded-md bg-white">
        <div class="flex flex-col items-center justify-center py-8">
            <i class="fas fa-spinner fa-spin text-3xl text-blue-600 mb-4"></i>
            <p class="text-gray-700 font-medium">Loading branch data...</p>
        </div>
    </div>
</div>

<!-- Success Alert Modal -->
<div id="successAlert" class="fixed inset-0 bg-gray-600 bg-opacity-50 overflow-y-auto h-full w-full hidden z-50 transition-opacity">
    <div class="relative top-20 mx-auto p-5 border w-full max-w-sm shadow-lg rounded-md bg-white transform transition-all">
        <div class="mt-3 text-center">
            <div class="mx-auto flex items-center justify-center h-12 w-12 rounded-full bg-green-100 mb-4">
                <i class="fas fa-check text-green-600 text-xl"></i>
            </div>
            <h3 class="text-lg font-medium text-gray-900 mb-2" id="successTitle">Success!</h3>
            <p class="text-sm text-gray-500 mb-4" id="successMessage">Operation completed successfully.</p>
            <div class="px-4 py-3">
                <button onclick="closeSuccessAlert()" class="w-full bg-green-600 hover:bg-green-700 text-white py-2 px-4 rounded-lg font-medium transition-colors">
                    Continue
                </button>
            </div>
        </div>
    </div>
</div>

<!-- Error Alert Modal -->
<div id="errorAlert" class="fixed inset-0 bg-gray-600 bg-opacity-50 overflow-y-auto h-full w-full hidden z-50 transition-opacity">
    <div class="relative top-20 mx-auto p-5 border w-full max-w-sm shadow-lg rounded-md bg-white transform transition-all">
        <div class="mt-3 text-center">
            <div class="mx-auto flex items-center justify-center h-12 w-12 rounded-full bg-red-100 mb-4">
                <i class="fas fa-exclamation-triangle text-red-600 text-xl"></i>
            </div>
            <h3 class="text-lg font-medium text-gray-900 mb-2" id="errorTitle">Error</h3>
            <p class="text-sm text-gray-500 mb-4" id="errorMessage">An error occurred.</p>
            <div class="px-4 py-3">
                <button onclick="closeErrorAlert()" class="w-full bg-red-600 hover:bg-red-700 text-white py-2 px-4 rounded-lg font-medium transition-colors">
                    Try Again
                </button>
            </div>
        </div>
    </div>
</div>

<!-- Confirmation Modal -->
<div id="confirmationModal" class="fixed inset-0 bg-gray-600 bg-opacity-50 overflow-y-auto h-full w-full hidden z-50 transition-opacity">
    <div class="relative top-20 mx-auto p-5 border w-full max-w-sm shadow-lg rounded-md bg-white transform transition-all">
        <div class="mt-3 text-center">
            <div class="mx-auto flex items-center justify-center h-12 w-12 rounded-full bg-yellow-100 mb-4">
                <i class="fas fa-question text-yellow-600 text-xl"></i>
            </div>
            <h3 class="text-lg font-medium text-gray-900 mb-2" id="confirmationTitle">Confirm Action</h3>
            <p class="text-sm text-gray-500 mb-4" id="confirmationMessage">Are you sure you want to proceed?</p>
            <div class="flex space-x-3 px-4 py-3">
                <button onclick="closeConfirmationModal(false)" class="flex-1 bg-gray-300 hover:bg-gray-400 text-gray-800 py-2 px-4 rounded-lg font-medium transition-colors">
                    Cancel
                </button>
                <button onclick="closeConfirmationModal(true)" class="flex-1 bg-blue-600 hover:bg-blue-700 text-white py-2 px-4 rounded-lg font-medium transition-colors">
                    Confirm
                </button>
            </div>
        </div>
    </div>
</div>
@endsection

@section('scripts')
<script>
    // Check if variables are already declared to avoid duplication
    if (typeof currentLoadingButton === 'undefined') {
        var currentLoadingButton = null;
    }
    if (typeof confirmationCallback === 'undefined') {
        var confirmationCallback = null;
    }

    // Get the base URL dynamically
    const baseUrl = '{{ url("/") }}';

    // Alert Modal Functions
    function showSuccessAlert(title, message) {
        const successTitle = document.getElementById('successTitle');
        const successMessage = document.getElementById('successMessage');
        const successAlert = document.getElementById('successAlert');
        
        if (successTitle && successMessage && successAlert) {
            successTitle.textContent = title;
            successMessage.textContent = message;
            successAlert.classList.remove('hidden');
        } else {
            console.error('Success alert elements not found');
            // Fallback to basic alert
            alert('SUCCESS: ' + title + ' - ' + message);
        }
    }

    function closeSuccessAlert() {
        const successAlert = document.getElementById('successAlert');
        if (successAlert) {
            successAlert.classList.add('hidden');
        }
    }

    function showErrorAlert(title, message) {
        const errorTitle = document.getElementById('errorTitle');
        const errorMessage = document.getElementById('errorMessage');
        const errorAlert = document.getElementById('errorAlert');
        
        if (errorTitle && errorMessage && errorAlert) {
            errorTitle.textContent = title;
            errorMessage.textContent = message;
            errorAlert.classList.remove('hidden');
        } else {
            console.error('Error alert elements not found');
            // Fallback to basic alert
            alert('ERROR: ' + title + ' - ' + message);
        }
    }

    function closeErrorAlert() {
        const errorAlert = document.getElementById('errorAlert');
        if (errorAlert) {
            errorAlert.classList.add('hidden');
        }
    }

    function showConfirmation(title, message, callback) {
        const confirmationTitle = document.getElementById('confirmationTitle');
        const confirmationMessage = document.getElementById('confirmationMessage');
        const confirmationModal = document.getElementById('confirmationModal');
        
        if (confirmationTitle && confirmationMessage && confirmationModal) {
            confirmationTitle.textContent = title;
            confirmationMessage.textContent = message;
            confirmationModal.classList.remove('hidden');
            confirmationCallback = callback;
        } else {
            console.error('Confirmation modal elements not found');
            // Fallback to basic confirm
            if (confirm(title + ': ' + message)) {
                callback();
            }
        }
    }

    function closeConfirmationModal(confirmed) {
        const confirmationModal = document.getElementById('confirmationModal');
        if (confirmationModal) {
            confirmationModal.classList.add('hidden');
        }
        if (confirmed && confirmationCallback) {
            confirmationCallback();
        }
        confirmationCallback = null;
    }

    // Modal functions
    function openCreateModal() {
        const createModal = document.getElementById('createModal');
        if (createModal) {
            createModal.classList.remove('hidden');
        }
    }

    function closeCreateModal() {
        const createModal = document.getElementById('createModal');
        if (createModal) {
            createModal.classList.add('hidden');
        }
        const createForm = document.getElementById('createBranchForm');
        if (createForm) {
            createForm.reset();
        }
    }

    function openEditModal(branchId) {
        console.log('Opening edit modal for branch:', branchId);
        
        // Store reference to the clicked button
        currentLoadingButton = event.target.closest('button');
        if (!currentLoadingButton) {
            showErrorAlert('Error', 'Unable to find the button element.');
            return;
        }
        
        const originalHTML = currentLoadingButton.innerHTML;
        currentLoadingButton.innerHTML = '<i class="fas fa-spinner fa-spin"></i>';
        currentLoadingButton.disabled = true;

        // Show loading overlay
        const editLoading = document.getElementById('editLoading');
        if (editLoading) {
            editLoading.classList.remove('hidden');
        }

        // Use the correct URL with the base path
        const url = `${baseUrl}/admin/branches/${branchId}`;

        fetch(url, {
            headers: {
                'Accept': 'application/json',
                'X-Requested-With': 'XMLHttpRequest'
            }
        })
        .then(response => {
            if (!response.ok) {
                throw new Error(`HTTP error! status: ${response.status}`);
            }
            return response.json();
        })
        .then(data => {
            if (!data.success) {
                throw new Error(data.message || 'Failed to load branch data');
            }

            // Populate the form with branch data
            const editBranchId = document.getElementById('edit_branch_id');
            const editBranchcode = document.getElementById('edit_branchcode');
            const editBranchname = document.getElementById('edit_branchname');
            const editSetupdate = document.getElementById('edit_setupdate');
            
            if (editBranchId) editBranchId.value = data.data.id;
            if (editBranchcode) editBranchcode.value = data.data.branchcode;
            if (editBranchname) editBranchname.value = data.data.branchname;
            if (editSetupdate) editSetupdate.value = data.data.setupdate;

            // Hide loading and show edit modal
            if (editLoading) editLoading.classList.add('hidden');
            const editModal = document.getElementById('editModal');
            if (editModal) editModal.classList.remove('hidden');
            
            // Restore button state
            currentLoadingButton.innerHTML = originalHTML;
            currentLoadingButton.disabled = false;
            currentLoadingButton = null;
        })
        .catch(error => {
            console.error('Error loading branch:', error);
            
            // Hide loading overlay
            if (editLoading) editLoading.classList.add('hidden');
            
            // Show error message
            showErrorAlert('Load Error', 'Failed to load branch data: ' + error.message);
            
            // Restore button state
            if (currentLoadingButton) {
                currentLoadingButton.innerHTML = originalHTML;
                currentLoadingButton.disabled = false;
                currentLoadingButton = null;
            }
        });
    }

    function closeEditModal() {
        const editModal = document.getElementById('editModal');
        if (editModal) {
            editModal.classList.add('hidden');
        }
        
        // Restore button state if still loading
        if (currentLoadingButton && currentLoadingButton.disabled) {
            currentLoadingButton.innerHTML = '<i class="fas fa-edit"></i>';
            currentLoadingButton.disabled = false;
            currentLoadingButton = null;
        }
    }

    // Form submissions
    document.addEventListener('DOMContentLoaded', function() {
        const createForm = document.getElementById('createBranchForm');
        const editForm = document.getElementById('editBranchForm');
        
        if (createForm) {
            createForm.addEventListener('submit', function(e) {
                e.preventDefault();
                
                const submitButton = this.querySelector('button[type="submit"]');
                const originalText = submitButton.innerHTML;
                
                submitButton.innerHTML = '<i class="fas fa-spinner fa-spin"></i> Creating...';
                submitButton.disabled = true;

                const formData = new FormData(this);
                
                fetch(`${baseUrl}/admin/branches`, {
                    method: 'POST',
                    body: formData,
                    headers: {
                        'X-CSRF-TOKEN': '{{ csrf_token() }}',
                        'Accept': 'application/json'
                    }
                })
                .then(response => response.json())
                .then(data => {
                    if (data.success) {
                        showSuccessAlert('Success', data.message);
                        closeCreateModal();
                        setTimeout(() => {
                            location.reload();
                        }, 1500);
                    } else {
                        showErrorAlert('Error', data.message);
                    }
                })
                .catch(error => {
                    console.error('Error:', error);
                    showErrorAlert('Error', 'Failed to create branch: ' + error.message);
                })
                .finally(() => {
                    submitButton.innerHTML = originalText;
                    submitButton.disabled = false;
                });
            });
        }
        
        if (editForm) {
            editForm.addEventListener('submit', function(e) {
                e.preventDefault();
                
                const branchId = document.getElementById('edit_branch_id')?.value;
                if (!branchId) {
                    showErrorAlert('Error', 'Branch ID not found.');
                    return;
                }
                
                const submitButton = this.querySelector('button[type="submit"]');
                const originalText = submitButton.innerHTML;
                
                submitButton.innerHTML = '<i class="fas fa-spinner fa-spin"></i> Updating...';
                submitButton.disabled = true;

                const formData = new FormData(this);
                
                fetch(`${baseUrl}/admin/branches/${branchId}`, {
                    method: 'POST',
                    body: formData,
                    headers: {
                        'X-CSRF-TOKEN': '{{ csrf_token() }}',
                        'X-HTTP-Method-Override': 'PUT',
                        'Accept': 'application/json'
                    }
                })
                .then(response => response.json())
                .then(data => {
                    if (data.success) {
                        showSuccessAlert('Success', data.message);
                        closeEditModal();
                        setTimeout(() => {
                            location.reload();
                        }, 1500);
                    } else {
                        showErrorAlert('Error', data.message);
                    }
                })
                .catch(error => {
                    console.error('Error:', error);
                    showErrorAlert('Error', 'Failed to update branch: ' + error.message);
                })
                .finally(() => {
                    submitButton.innerHTML = originalText;
                    submitButton.disabled = false;
                });
            });
        }
    });

    function toggleStatus(branchId) {
        showConfirmation(
            'Change Branch Status',
            'Are you sure you want to change the status of this branch?',
            () => {
                const button = event.target.closest('button');
                const originalHTML = button.innerHTML;
                button.innerHTML = '<i class="fas fa-spinner fa-spin"></i>';
                button.disabled = true;

                fetch(`${baseUrl}/admin/branches/${branchId}/toggle-status`, {
                    method: 'POST',
                    headers: {
                        'X-CSRF-TOKEN': '{{ csrf_token() }}',
                        'Content-Type': 'application/json',
                        'Accept': 'application/json'
                    }
                })
                .then(response => response.json())
                .then(data => {
                    if (data.success) {
                        showSuccessAlert('Success', data.message);
                        setTimeout(() => {
                            location.reload();
                        }, 1500);
                    } else {
                        showErrorAlert('Error', data.message);
                    }
                })
                .catch(error => {
                    console.error('Error:', error);
                    showErrorAlert('Error', 'Failed to update branch status: ' + error.message);
                })
                .finally(() => {
                    button.innerHTML = originalHTML;
                    button.disabled = false;
                });
            }
        );
    }

    function confirmDelete(branchId) {
        showConfirmation(
            'Delete Branch',
            'Are you sure you want to delete this branch? This action cannot be undone.',
            () => {
                const button = event.target.closest('button');
                const originalHTML = button.innerHTML;
                button.innerHTML = '<i class="fas fa-spinner fa-spin"></i>';
                button.disabled = true;

                fetch(`${baseUrl}/admin/branches/${branchId}`, {
                    method: 'DELETE',
                    headers: {
                        'X-CSRF-TOKEN': '{{ csrf_token() }}',
                        'Content-Type': 'application/json',
                        'Accept': 'application/json'
                    }
                })
                .then(response => response.json())
                .then(data => {
                    if (data.success) {
                        showSuccessAlert('Success', data.message);
                        setTimeout(() => {
                            location.reload();
                        }, 1500);
                    } else {
                        showErrorAlert('Error', data.message);
                    }
                })
                .catch(error => {
                    console.error('Error:', error);
                    showErrorAlert('Error', 'Failed to delete branch: ' + error.message);
                })
                .finally(() => {
                    button.innerHTML = originalHTML;
                    button.disabled = false;
                });
            }
        );
    }

    // Close modals when clicking outside
    document.addEventListener('click', function(event) {
        const modals = [
            'createModal', 'editModal', 'editLoading', 
            'successAlert', 'errorAlert', 'confirmationModal'
        ];
        
        modals.forEach(modalId => {
            const modal = document.getElementById(modalId);
            if (event.target === modal) {
                if (modalId === 'successAlert') closeSuccessAlert();
                else if (modalId === 'errorAlert') closeErrorAlert();
                else if (modalId === 'confirmationModal') closeConfirmationModal(false);
                else if (modalId === 'editLoading') {
                    modal.classList.add('hidden');
                    closeEditModal();
                }
                else if (modalId === 'createModal') closeCreateModal();
                else if (modalId === 'editModal') closeEditModal();
            }
        });
    });

    // Close modals with Escape key
    document.addEventListener('keydown', function(event) {
        if (event.key === 'Escape') {
            closeCreateModal();
            closeEditModal();
            closeSuccessAlert();
            closeErrorAlert();
            closeConfirmationModal(false);
            const editLoading = document.getElementById('editLoading');
            if (editLoading) editLoading.classList.add('hidden');
        }
    });
</script>
@endsection