@php
    // Define page variables
    $pageTitle = 'Loan Details';
    $pageSubtitle = 'View detailed information about this loan';
@endphp

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>{{ $pageTitle }} | IMMIA FINANCE</title>
    
    <script src="https://cdn.tailwindcss.com"></script>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    
    <style>
        /* --- ZEDBANKOS THEME COLORS --- */
        :root {
            --zedbank-blue: #0077C5; /* ThemeBlue */
            --zedbank-blue-dark: #005A94; /* Darker shade for hover/active background */
        }

        /* Custom Tailwind-like classes */
        .bg-zedbank-blue { background-color: var(--zedbank-blue); }
        .bg-zedbank-blue-dark { background-color: var(--zedbank-blue-dark); }
        .text-zedbank-blue { color: var(--zedbank-blue); }
        .hover\:bg-zedbank-blue-dark:hover { background-color: var(--zedbank-blue-dark); }

        /* Sidebar/Main Content Layout */
        .lg\:ml-64 { margin-left: 16rem; }
        @media (max-width: 1024px) { 
            .lg\:ml-64 { margin-left: 0 !important; }
        }

        /* Local Overrides */
        .focus\:ring-primary:focus { --tw-ring-color: var(--zedbank-blue); }
        .focus\:border-primary:focus { border-color: var(--zedbank-blue); }
        .bg-primary { background-color: var(--zedbank-blue); }
        
        /* Custom colors for balances */
        .text-interest { color: #8B4513; }
        .bg-interest-light { background-color: #FFF8DC; }
        .text-penalty { color: #FF6B35; }
        .bg-penalty-light { background-color: #FFE5D9; }
        .text-processing { color: #9932CC; }
        .bg-processing-light { background-color: #E6E6FA; }
        .text-total { color: #000080; }
        .bg-total-light { background-color: #F0F8FF; }
    </style>
</head>
<body class="bg-gray-50 min-h-screen">
    
    {{-- Assuming 'accounting.partials.sidebar' exists --}}
    @include('accounting.partials.sidebar')

    <div class="lg:ml-64 min-h-screen">
        
        {{-- Pass page variables to the header partial --}}
        @include('accounting.partials.header', [
            'title' => $pageTitle,
            'subtitle' => $pageSubtitle
        ])

        <main class="p-6">
            {{-- Page Header with Actions --}}
            <div class="mb-8">
                <div class="flex flex-col md:flex-row md:items-center justify-between gap-4">
                    <div>
                        <div class="flex items-center mb-2">
                            <a href="{{ route('admin.loans.active') }}" 
                               class="inline-flex items-center text-zedbank-blue hover:text-zedbank-blue-dark mr-3 transition">
                                <i class="fas fa-arrow-left mr-1"></i>
                                Back to Loans
                            </a>
                            <h1 class="text-3xl font-bold text-gray-800">Loan: {{ $loan->loannumber }}</h1>
                        </div>
                        <p class="text-gray-600">Detailed view of loan information, balances, and transactions.</p>
                    </div>
                    
                    <div class="flex flex-wrap gap-2">
                        {{-- Record Payment Button --}}
                        <a href="{{ route('admin.loans.payment-form', $loan->loanid) }}" 
                           class="px-4 py-2 bg-green-600 text-white rounded-lg hover:bg-green-700 transition font-medium flex items-center">
                            <i class="fas fa-dollar-sign mr-2"></i>
                            Record Payment
                        </a>
                        
                        {{-- NPL Form Button (Only for active loans) --}}
                        @if($loan->status === 'active')
                            <a href="{{ route('admin.loans.npl-form', $loan->loanid) }}" 
                               class="px-4 py-2 bg-red-600 text-white rounded-lg hover:bg-red-700 transition font-medium flex items-center">
                                <i class="fas fa-exclamation-triangle mr-2"></i>
                                Classify as NPL
                            </a>
                        @endif
                        
                        {{-- Close Loan Button (Only for active loans) --}}
                        @if($loan->status === 'active')
                            <a href="{{ route('admin.loans.close-form', $loan->loanid) }}" 
                               class="px-4 py-2 bg-blue-600 text-white rounded-lg hover:bg-blue-700 transition font-medium flex items-center">
                                <i class="fas fa-lock mr-2"></i>
                                Close Loan
                            </a>
                        @endif
                        
                        {{-- Reopen/Reverse Actions (For closed/NPL loans) --}}
                        @if($loan->status === 'closed')
                            <form method="POST" action="{{ route('admin.loans.reverse', $loan->loanid) }}" 
                                  class="inline" 
                                  onsubmit="return confirm('Are you sure you want to reopen this closed loan?')">
                                @csrf
                                <button type="submit" 
                                        class="px-4 py-2 bg-yellow-600 text-white rounded-lg hover:bg-yellow-700 transition font-medium flex items-center">
                                    <i class="fas fa-undo mr-2"></i>
                                    Reopen Loan
                                </button>
                            </form>
                        @endif
                        
                        @if($loan->status === 'default')
                            <form method="POST" action="{{ route('admin.loans.reverse', $loan->loanid) }}" 
                                  class="inline" 
                                  onsubmit="return confirm('Are you sure you want to restore this loan to active status?')">
                                @csrf
                                <button type="submit" 
                                        class="px-4 py-2 bg-green-600 text-white rounded-lg hover:bg-green-700 transition font-medium flex items-center">
                                    <i class="fas fa-check-circle mr-2"></i>
                                    Restore to Active
                                </button>
                            </form>
                        @endif
                    </div>
                </div>
            </div>
            
            {{-- Session Messages --}}
            @if(session('success'))
                <div class="mb-6 bg-green-50 border-l-4 border-green-500 p-4">
                    <div class="flex">
                        <div class="flex-shrink-0">
                            <i class="fas fa-check-circle text-green-500 text-xl"></i>
                        </div>
                        <div class="ml-3">
                            <p class="text-green-700 font-medium">{{ session('success') }}</p>
                        </div>
                    </div>
                </div>
            @endif
            
            @if(session('error'))
                <div class="mb-6 bg-red-50 border-l-4 border-red-500 p-4">
                    <div class="flex">
                        <div class="flex-shrink-0">
                            <i class="fas fa-exclamation-circle text-red-500 text-xl"></i>
                        </div>
                        <div class="ml-3">
                            <p class="text-red-700 font-medium">{{ session('error') }}</p>
                        </div>
                    </div>
                </div>
            @endif
            
            {{-- Quick Stats Cards --}}
            <div class="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-5 gap-4 mb-6">
                <div class="bg-white rounded-lg p-4 shadow-sm border">
                    <div class="flex items-center justify-between">
                        <div>
                            <p class="text-sm text-gray-600">Loan Status</p>
                            @php
                                $statusBadgeClass = '';
                                if ($loan->status === 'active') $statusBadgeClass = 'bg-green-100 text-green-800';
                                elseif ($loan->status === 'closed') $statusBadgeClass = 'bg-gray-100 text-gray-800';
                                elseif ($loan->status === 'default') $statusBadgeClass = 'bg-red-100 text-red-800';
                                else $statusBadgeClass = 'bg-blue-100 text-blue-800';
                            @endphp
                            <span class="inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium {{ $statusBadgeClass }}">
                                <i class="fas fa-{{ $loan->status === 'active' ? 'check' : ($loan->status === 'closed' ? 'lock' : 'exclamation-triangle') }} mr-1"></i>
                                {{ ucfirst($loan->status) }}
                            </span>
                        </div>
                        <div class="w-10 h-10 {{ $loan->status === 'active' ? 'bg-green-100' : ($loan->status === 'closed' ? 'bg-gray-100' : 'bg-red-100') }} rounded-lg flex items-center justify-center">
                            <i class="fas fa-chart-line {{ $loan->status === 'active' ? 'text-green-600' : ($loan->status === 'closed' ? 'text-gray-600' : 'text-red-600') }}"></i>
                        </div>
                    </div>
                </div>
                
                <div class="bg-white rounded-lg p-4 shadow-sm border">
                    <div class="flex items-center justify-between">
                        <div>
                            <p class="text-sm text-gray-600">Principal Balance</p>
                            <p class="text-lg font-bold text-red-600">K {{ number_format($loan->principalbalance, 2) }}</p>
                        </div>
                        <div class="w-10 h-10 bg-red-100 rounded-lg flex items-center justify-center">
                            <i class="fas fa-home text-red-600"></i>
                        </div>
                    </div>
                </div>
                
                <div class="bg-white rounded-lg p-4 shadow-sm border">
                    <div class="flex items-center justify-between">
                        <div>
                            <p class="text-sm text-gray-600">Interest Balance</p>
                            <p class="text-lg font-bold text-interest">K {{ number_format($loan->interestbalance, 2) }}</p>
                        </div>
                        <div class="w-10 h-10 bg-interest-light rounded-lg flex items-center justify-center">
                            <i class="fas fa-chart-line text-interest"></i>
                        </div>
                    </div>
                </div>
                
                <div class="bg-white rounded-lg p-4 shadow-sm border">
                    <div class="flex items-center justify-between">
                        <div>
                            <p class="text-sm text-gray-600">Penalty Balance</p>
                            <p class="text-lg font-bold text-penalty">K {{ number_format($loan->penalty_balance, 2) }}</p>
                        </div>
                        <div class="w-10 h-10 bg-penalty-light rounded-lg flex items-center justify-center">
                            <i class="fas fa-exclamation-circle text-penalty"></i>
                        </div>
                    </div>
                </div>
                
                <div class="bg-white rounded-lg p-4 shadow-sm border">
                    <div class="flex items-center justify-between">
                        <div>
                            <p class="text-sm text-gray-600">Total Outstanding</p>
                            <p class="text-lg font-bold text-total">K {{ number_format($loan->totalbalance, 2) }}</p>
                        </div>
                        <div class="w-10 h-10 bg-total-light rounded-lg flex items-center justify-center">
                            <i class="fas fa-balance-scale text-total"></i>
                        </div>
                    </div>
                </div>
            </div>
            
            {{-- Main Content Grid --}}
            <div class="grid grid-cols-1 lg:grid-cols-3 gap-6">
                {{-- Left Column: Loan Details --}}
                <div class="lg:col-span-2 space-y-6">
                    {{-- Loan Information Card --}}
                    <div class="bg-white rounded-lg shadow-sm border border-gray-200 overflow-hidden">
                        <div class="border-b border-gray-200 px-6 py-4">
                            <h2 class="text-xl font-semibold text-gray-800">Loan Information</h2>
                        </div>
                        <div class="p-6">
                            <div class="grid grid-cols-1 md:grid-cols-2 gap-6">
                                {{-- Column 1 --}}
                                <div class="space-y-4">
                                    <div>
                                        <p class="text-sm text-gray-600">Loan Number</p>
                                        <p class="font-bold text-gray-800">{{ $loan->loannumber }}</p>
                                    </div>
                                    
                                    <div>
                                        <p class="text-sm text-gray-600">Customer</p>
                                        <p class="font-bold text-gray-800">
                                            @if($loan->customer)
                                                {{ $loan->customer->first_name }} {{ $loan->customer->surname }}
                                                <span class="text-xs text-gray-500 block">{{ $loan->customer->customer_number }}</span>
                                            @else
                                                <span class="text-gray-500">—</span>
                                            @endif
                                        </p>
                                    </div>
                                    
                                    <div>
                                        <p class="text-sm text-gray-600">Loan Product</p>
                                        <p class="font-bold text-gray-800">
                                            {{ $loan->loanType->product ?? 'N/A' }}
                                        </p>
                                    </div>
                                    
                                    <div>
                                        <p class="text-sm text-gray-600">Original Amount</p>
                                        <p class="font-bold text-gray-800">K {{ number_format($loan->amount, 2) }}</p>
                                    </div>
                                </div>
                                
                                {{-- Column 2 --}}
                                <div class="space-y-4">
                                    <div>
                                        <p class="text-sm text-gray-600">Disbursement Date</p>
                                        <p class="font-bold text-gray-800">
                                            {{ $loan->disburseddate ? \Carbon\Carbon::parse($loan->disburseddate)->format('M d, Y') : 'N/A' }}
                                        </p>
                                    </div>
                                    
                                    <div>
                                        <p class="text-sm text-gray-600">Maturity Date</p>
                                        <p class="font-bold text-gray-800">
                                            {{ $loan->maturitydate ? \Carbon\Carbon::parse($loan->maturitydate)->format('M d, Y') : 'N/A' }}
                                        </p>
                                    </div>
                                    
                                    <div>
                                        <p class="text-sm text-gray-600">Interest Rate</p>
                                        <p class="font-bold text-gray-800">{{ number_format($loan->interestrate, 2) }}%</p>
                                        <p class="text-xs text-gray-500">
                                            {{ $loan->interest_method === 'simple_interest' ? 'Simple Interest' : 'Reducing Balance' }}
                                        </p>
                                    </div>
                                    
                                    <div>
                                        <p class="text-sm text-gray-600">Loan Term</p>
                                        <p class="font-bold text-gray-800">
                                            {{ $loan->loanterm }} 
                                            @if($loan->paymentfrequency === 'weekly')
                                                weeks
                                            @elseif($loan->paymentfrequency === 'monthly')
                                                months
                                            @else
                                                days
                                            @endif
                                        </p>
                                        <p class="text-xs text-gray-500">
                                            {{ ucfirst($loan->paymentfrequency) }} payments
                                        </p>
                                    </div>
                                </div>
                            </div>
                            
                            {{-- Additional Details Section --}}
                            @if($loan->disbursement_method || $loan->payment_account_type || $loan->disbursement_notes)
                                <div class="mt-6 pt-6 border-t border-gray-200">
                                    <h3 class="text-lg font-semibold text-gray-800 mb-3">Disbursement Details</h3>
                                    <div class="grid grid-cols-1 md:grid-cols-3 gap-4">
                                        @if($loan->disbursement_method)
                                            <div>
                                                <p class="text-sm text-gray-600">Disbursement Method</p>
                                                <p class="font-medium text-gray-800">{{ $loan->disbursement_method }}</p>
                                            </div>
                                        @endif
                                        
                                        @if($loan->payment_account_type)
                                            <div>
                                                <p class="text-sm text-gray-600">Payment Account</p>
                                                <p class="font-medium text-gray-800">{{ $loan->payment_account_type }}</p>
                                            </div>
                                        @endif
                                        
                                        @if($loan->disbursement_notes)
                                            <div class="md:col-span-3">
                                                <p class="text-sm text-gray-600">Disbursement Notes</p>
                                                <p class="font-medium text-gray-800">{{ $loan->disbursement_notes }}</p>
                                            </div>
                                        @endif
                                    </div>
                                </div>
                            @endif
                            
                            {{-- Closure/NPL Details Section --}}
                            @if($loan->status === 'closed' || $loan->status === 'default')
                                <div class="mt-6 pt-6 border-t border-gray-200">
                                    <h3 class="text-lg font-semibold text-gray-800 mb-3">
                                        @if($loan->status === 'closed')
                                            Closure Details
                                        @else
                                            NPL Classification Details
                                        @endif
                                    </h3>
                                    
                                    @if($loan->status === 'closed')
                                        <div class="grid grid-cols-1 md:grid-cols-2 gap-4">
                                            <div>
                                                <p class="text-sm text-gray-600">Closed Date</p>
                                                <p class="font-medium text-gray-800">
                                                    {{ $loan->closeddate ? \Carbon\Carbon::parse($loan->closeddate)->format('M d, Y') : 'N/A' }}
                                                </p>
                                            </div>
                                            
                                            @if($loan->closure_reason)
                                                <div>
                                                    <p class="text-sm text-gray-600">Closure Reason</p>
                                                    <p class="font-medium text-gray-800">
                                                        @php
                                                            $closureReasons = [
                                                                'fully_settled' => 'Fully Settled',
                                                                'loan_forgiveness' => 'Loan Forgiveness',
                                                                'settlement_agreement' => 'Settlement Agreement',
                                                                'write_off_approved' => 'Write-off Approved',
                                                                'customer_bankruptcy' => 'Customer Bankruptcy',
                                                                'deceased_customer' => 'Deceased Customer',
                                                                'system_error' => 'System Error Correction',
                                                                'other' => 'Other'
                                                            ];
                                                            echo $closureReasons[$loan->closure_reason] ?? $loan->closure_reason;
                                                        @endphp
                                                    </p>
                                                </div>
                                            @endif
                                            
                                            @if($loan->closure_settlement_amount)
                                                <div>
                                                    <p class="text-sm text-gray-600">Final Settlement</p>
                                                    <p class="font-medium text-gray-800">
                                                        K {{ number_format($loan->closure_settlement_amount, 2) }}
                                                    </p>
                                                </div>
                                            @endif
                                            
                                            @if($loan->closure_notes)
                                                <div class="md:col-span-2">
                                                    <p class="text-sm text-gray-600">Closure Notes</p>
                                                    <p class="font-medium text-gray-800">{{ $loan->closure_notes }}</p>
                                                </div>
                                            @endif
                                        </div>
                                    @endif
                                    
                                    @if($loan->status === 'default')
                                        <div class="grid grid-cols-1 md:grid-cols-2 gap-4">
                                            @if($loan->npl_classification_date)
                                                <div>
                                                    <p class="text-sm text-gray-600">NPL Classification Date</p>
                                                    <p class="font-medium text-gray-800">
                                                        {{ \Carbon\Carbon::parse($loan->npl_classification_date)->format('M d, Y') }}
                                                    </p>
                                                </div>
                                            @endif
                                            
                                            @if($loan->npl_reason)
                                                <div>
                                                    <p class="text-sm text-gray-600">NPL Reason</p>
                                                    <p class="font-medium text-gray-800">
                                                        @php
                                                            $nplReasons = [
                                                                'payment_default' => 'Payment Default',
                                                                'financial_difficulties' => 'Financial Difficulties',
                                                                'bankruptcy' => 'Bankruptcy',
                                                                'breach_of_covenants' => 'Breach of Covenants',
                                                                'significant_doubt' => 'Significant Doubt',
                                                                'legal_issues' => 'Legal Issues',
                                                                'death_disability' => 'Death/Disability',
                                                                'other' => 'Other'
                                                            ];
                                                            echo $nplReasons[$loan->npl_reason] ?? $loan->npl_reason;
                                                        @endphp
                                                    </p>
                                                </div>
                                            @endif
                                            
                                            @if($loan->npl_category)
                                                <div>
                                                    <p class="text-sm text-gray-600">NPL Category</p>
                                                    <p class="font-medium text-gray-800">
                                                        @php
                                                            $nplCategories = [
                                                                'substandard' => 'Substandard',
                                                                'doubtful' => 'Doubtful',
                                                                'loss' => 'Loss',
                                                                'watch' => 'Watch List'
                                                            ];
                                                            echo $nplCategories[$loan->npl_category] ?? $loan->npl_category;
                                                        @endphp
                                                    </p>
                                                </div>
                                            @endif
                                            
                                            @if($loan->provision_percentage)
                                                <div>
                                                    <p class="text-sm text-gray-600">Provision Percentage</p>
                                                    <p class="font-medium text-gray-800">
                                                        {{ number_format($loan->provision_percentage, 2) }}%
                                                        @if($loan->provision_amount)
                                                            <span class="text-xs text-gray-500 block">
                                                                Amount: K {{ number_format($loan->provision_amount, 2) }}
                                                            </span>
                                                        @endif
                                                    </p>
                                                </div>
                                            @endif
                                            
                                            @if($loan->assessment_details)
                                                <div class="md:col-span-2">
                                                    <p class="text-sm text-gray-600">Assessment Details</p>
                                                    <p class="font-medium text-gray-800">{{ $loan->assessment_details }}</p>
                                                </div>
                                            @endif
                                        </div>
                                    @endif
                                </div>
                            @endif
                        </div>
                    </div>
                    
                    {{-- Payment History Card --}}
                    <div class="bg-white rounded-lg shadow-sm border border-gray-200 overflow-hidden">
                        <div class="border-b border-gray-200 px-6 py-4 flex justify-between items-center">
                            <h2 class="text-xl font-semibold text-gray-800">Payment History</h2>
                            <span class="text-sm text-gray-600">
                                {{ $loan->repayments->count() }} payment(s)
                            </span>
                        </div>
                        <div class="p-6">
                            @if($loan->repayments->count() > 0)
                                <div class="overflow-x-auto">
                                    <table class="min-w-full divide-y divide-gray-200">
                                        <thead>
                                            <tr>
                                                <th class="px-4 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Date</th>
                                                <th class="px-4 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Reference</th>
                                                <th class="px-4 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Method</th>
                                                <th class="px-4 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Principal</th>
                                                <th class="px-4 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Interest</th>
                                                <th class="px-4 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Penalties</th>
                                                <th class="px-4 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Processing</th>
                                                <th class="px-4 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Total</th>
                                                <th class="px-4 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Status</th>
                                                <th class="px-4 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Actions</th>
                                            </tr>
                                        </thead>
                                        <tbody class="divide-y divide-gray-200">
                                            @foreach($loan->repayments->sortByDesc('rdate') as $repayment)
                                                <tr class="hover:bg-gray-50">
                                                    <td class="px-4 py-3 text-sm text-gray-900">
                                                        {{ \Carbon\Carbon::parse($repayment->rdate)->format('M d, Y') }}
                                                        @if($repayment->posted_date)
                                                            <br>
                                                            <span class="text-xs text-gray-500">
                                                                Posted: {{ \Carbon\Carbon::parse($repayment->posted_date)->format('M d, Y') }}
                                                            </span>
                                                        @endif
                                                    </td>
                                                    <td class="px-4 py-3 text-sm text-gray-900">
                                                        {{ $repayment->paymentreference ?? 'N/A' }}
                                                    </td>
                                                    <td class="px-4 py-3 text-sm text-gray-900">
                                                        {{ $repayment->pmethod }}
                                                    </td>
                                                    <td class="px-4 py-3 text-sm font-medium {{ $repayment->rprincipal > 0 ? 'text-green-700' : 'text-gray-500' }}">
                                                        K {{ number_format($repayment->rprincipal, 2) }}
                                                    </td>
                                                    <td class="px-4 py-3 text-sm font-medium {{ $repayment->rinterest > 0 ? 'text-interest' : 'text-gray-500' }}">
                                                        K {{ number_format($repayment->rinterest, 2) }}
                                                    </td>
                                                    <td class="px-4 py-3 text-sm font-medium {{ $repayment->penalties > 0 ? 'text-red-600' : 'text-gray-500' }}">
                                                        K {{ number_format($repayment->penalties, 2) }}
                                                    </td>
                                                    <td class="px-4 py-3 text-sm font-medium {{ $repayment->processing_fees_amount_paid > 0 ? 'text-purple-600' : 'text-gray-500' }}">
                                                        K {{ number_format($repayment->processing_fees_amount_paid, 2) }}
                                                    </td>
                                                    <td class="px-4 py-3 text-sm font-bold text-gray-900">
                                                        K {{ number_format($repayment->rtotal, 2) }}
                                                    </td>
                                                    <td class="px-4 py-3">
                                                        @php
                                                            $statusBadgeClass = '';
                                                            $statusIcon = '';
                                                            
                                                            if ($repayment->status === 'posted') {
                                                                $statusBadgeClass = 'bg-green-100 text-green-800';
                                                                $statusIcon = 'fa-check-circle';
                                                            } elseif ($repayment->status === 'pending') {
                                                                $statusBadgeClass = 'bg-yellow-100 text-yellow-800';
                                                                $statusIcon = 'fa-clock';
                                                            } elseif ($repayment->status === 'reversed') {
                                                                $statusBadgeClass = 'bg-red-100 text-red-800';
                                                                $statusIcon = 'fa-undo';
                                                            } elseif ($repayment->status === 'failed') {
                                                                $statusBadgeClass = 'bg-red-100 text-red-800';
                                                                $statusIcon = 'fa-times-circle';
                                                            } else {
                                                                $statusBadgeClass = 'bg-gray-100 text-gray-800';
                                                                $statusIcon = 'fa-question-circle';
                                                            }
                                                        @endphp
                                                        <span class="inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium {{ $statusBadgeClass }}">
                                                            <i class="fas {{ $statusIcon }} mr-1"></i>
                                                            {{ ucfirst($repayment->status) }}
                                                        </span>
                                                    </td>
                                                    <td class="px-4 py-3 text-sm">
                                                        <div class="flex space-x-2">
                                                            @if($repayment->status === 'posted')
                                                                <a href="{{ route('admin.repayments.show', $repayment->repaymentid) }}" 
                                                                   class="text-blue-600 hover:text-blue-900" 
                                                                   title="View Details">
                                                                    <i class="fas fa-eye"></i>
                                                                </a>
                                                                @if(auth()->user()->hasPermission('payments.reverse'))
                                                                    <form action="{{ route('admin.repayments.reverse', $repayment->repaymentid) }}" 
                                                                          method="POST" 
                                                                          onsubmit="return confirm('Are you sure you want to reverse this payment?')"
                                                                          class="inline">
                                                                        @csrf
                                                                        @method('POST')
                                                                        <button type="submit" 
                                                                                class="text-red-600 hover:text-red-900"
                                                                                title="Reverse Payment">
                                                                            <i class="fas fa-undo"></i>
                                                                        </button>
                                                                    </form>
                                                                @endif
                                                            @endif
                                                        </div>
                                                    </td>
                                                </tr>
                                            @endforeach
                                        </tbody>
                                        {{-- Summary Footer --}}
                                        <tfoot class="bg-gray-50">
                                            <tr>
                                                <td colspan="3" class="px-4 py-3 text-sm font-medium text-gray-900 text-right">
                                                    Totals:
                                                </td>
                                                <td class="px-4 py-3 text-sm font-bold text-green-700">
                                                    K {{ number_format($loan->repayments->sum('rprincipal'), 2) }}
                                                </td>
                                                <td class="px-4 py-3 text-sm font-bold text-interest">
                                                    K {{ number_format($loan->repayments->sum('rinterest'), 2) }}
                                                </td>
                                                <td class="px-4 py-3 text-sm font-bold text-red-600">
                                                    K {{ number_format($loan->repayments->sum('penalties'), 2) }}
                                                </td>
                                                <td class="px-4 py-3 text-sm font-bold text-purple-600">
                                                    K {{ number_format($loan->repayments->sum('processing_fees_amount_paid'), 2) }}
                                                </td>
                                                <td class="px-4 py-3 text-sm font-bold text-gray-900">
                                                    K {{ number_format($loan->repayments->sum('rtotal'), 2) }}
                                                </td>
                                                <td colspan="2"></td>
                                            </tr>
                                        </tfoot>
                                    </table>
                                </div>
                            @else
                                <div class="text-center py-8">
                                    <i class="fas fa-money-check-alt text-4xl text-gray-300 mb-4"></i>
                                    <p class="text-gray-500">No payment history available</p>
                                    <a href="{{ route('admin.loans.payment-form', $loan->loanid) }}" 
                                       class="mt-4 inline-flex items-center px-4 py-2 bg-zedbank-blue text-white rounded-lg hover:bg-zedbank-blue-dark transition">
                                        <i class="fas fa-dollar-sign mr-2"></i>
                                        Record First Payment
                                    </a>
                                </div>
                            @endif
                        </div>
                    </div>
                </div>
                
                {{-- Right Column: Actions & Summary --}}
                <div class="space-y-6">
                    {{-- Quick Actions Card --}}
                    <div class="bg-white rounded-lg shadow-sm border border-gray-200 overflow-hidden">
                        <div class="border-b border-gray-200 px-6 py-4">
                            <h2 class="text-xl font-semibold text-gray-800">Quick Actions</h2>
                        </div>
                        <div class="p-6">
                            <div class="space-y-3">
                                {{-- Record Payment Button --}}
                                <a href="{{ route('admin.loans.payment-form', $loan->loanid) }}" 
                                   class="w-full flex items-center justify-between px-4 py-3 bg-green-50 hover:bg-green-100 border border-green-200 rounded-lg transition group">
                                    <div class="flex items-center">
                                        <div class="w-8 h-8 bg-green-100 rounded-lg flex items-center justify-center mr-3">
                                            <i class="fas fa-dollar-sign text-green-600"></i>
                                        </div>
                                        <div>
                                            <p class="font-medium text-green-800">Record Payment</p>
                                            <p class="text-xs text-green-600">Process a new payment</p>
                                        </div>
                                    </div>
                                    <i class="fas fa-chevron-right text-green-400 group-hover:text-green-600"></i>
                                </a>
                                
                                {{-- Classify as NPL Button (Only for active loans) --}}
                                @if($loan->status === 'active')
                                    <a href="{{ route('admin.loans.npl-form', $loan->loanid) }}" 
                                       class="w-full flex items-center justify-between px-4 py-3 bg-red-50 hover:bg-red-100 border border-red-200 rounded-lg transition group">
                                        <div class="flex items-center">
                                            <div class="w-8 h-8 bg-red-100 rounded-lg flex items-center justify-center mr-3">
                                                <i class="fas fa-exclamation-triangle text-red-600"></i>
                                            </div>
                                            <div>
                                                <p class="font-medium text-red-800">Classify as NPL</p>
                                                <p class="text-xs text-red-600">Mark as non-performing</p>
                                            </div>
                                        </div>
                                        <i class="fas fa-chevron-right text-red-400 group-hover:text-red-600"></i>
                                    </a>
                                @endif
                                
                                {{-- Close Loan Button (Only for active loans) --}}
                                @if($loan->status === 'active')
                                    <a href="{{ route('admin.loans.close-form', $loan->loanid) }}" 
                                       class="w-full flex items-center justify-between px-4 py-3 bg-blue-50 hover:bg-blue-100 border border-blue-200 rounded-lg transition group">
                                        <div class="flex items-center">
                                            <div class="w-8 h-8 bg-blue-100 rounded-lg flex items-center justify-center mr-3">
                                                <i class="fas fa-lock text-blue-600"></i>
                                            </div>
                                            <div>
                                                <p class="font-medium text-blue-800">Close Loan</p>
                                                <p class="text-xs text-blue-600">Permanently close this loan</p>
                                            </div>
                                        </div>
                                        <i class="fas fa-chevron-right text-blue-400 group-hover:text-blue-600"></i>
                                    </a>
                                @endif
                                
                                {{-- Add this button to your show.blade.php in the action buttons section --}}
                                <a href="{{ route('admin.loans.loan-statement', $loan->loanid) }}" 
                                   class="px-4 py-2 bg-purple-600 text-white rounded-lg hover:bg-purple-700 transition font-medium flex items-center">
                                   <i class="fas fa-file-invoice-dollar mr-2"></i>
                                      View Statement
                                </a>
                                
                                {{-- View Amortization Schedule --}}
                                @if($loan->paymentSchedules && $loan->paymentSchedules->count() > 0)
                                    <a href="{{ route('admin.loans.schedule', $loan->loanid) }}" 
                                       class="w-full flex items-center justify-between px-4 py-3 bg-purple-50 hover:bg-purple-100 border border-purple-200 rounded-lg transition group">
                                        <div class="flex items-center">
                                            <div class="w-8 h-8 bg-purple-100 rounded-lg flex items-center justify-center mr-3">
                                                <i class="fas fa-calendar-alt text-purple-600"></i>
                                            </div>
                                            <div>
                                                <p class="font-medium text-purple-800">View Schedule</p>
                                                <p class="text-xs text-purple-600">Payment amortization</p>
                                            </div>
                                        </div>
                                        <i class="fas fa-chevron-right text-purple-400 group-hover:text-purple-600"></i>
                                    </a>
                                @endif
                                
                                {{-- Export Loan Details --}}
                                <button onclick="alert('Export feature coming soon!')" 
                                        class="w-full flex items-center justify-between px-4 py-3 bg-gray-50 hover:bg-gray-100 border border-gray-200 rounded-lg transition group">
                                    <div class="flex items-center">
                                        <div class="w-8 h-8 bg-gray-100 rounded-lg flex items-center justify-center mr-3">
                                            <i class="fas fa-download text-gray-600"></i>
                                        </div>
                                        <div>
                                            <p class="font-medium text-gray-800">Export Details</p>
                                            <p class="text-xs text-gray-600">Download loan report</p>
                                        </div>
                                    </div>
                                    <i class="fas fa-chevron-right text-gray-400 group-hover:text-gray-600"></i>
                                </button>
                           
                           
                            </div>
                        </div>
                    </div>
                    
                    {{-- Balance Summary Card --}}
                    <div class="bg-white rounded-lg shadow-sm border border-gray-200 overflow-hidden">
                        <div class="border-b border-gray-200 px-6 py-4">
                            <h2 class="text-xl font-semibold text-gray-800">Balance Summary</h2>
                        </div>
                        <div class="p-6">
                            <dl class="space-y-3">
                                <div class="flex justify-between items-center">
                                    <dt class="text-sm text-gray-600">Principal Balance</dt>
                                    <dd class="text-sm font-bold text-red-600">K {{ number_format($loan->principalbalance, 2) }}</dd>
                                </div>
                                <div class="flex justify-between items-center">
                                    <dt class="text-sm text-gray-600">Interest Balance</dt>
                                    <dd class="text-sm font-bold text-interest">K {{ number_format($loan->interestbalance, 2) }}</dd>
                                </div>
                                <div class="flex justify-between items-center">
                                    <dt class="text-sm text-gray-600">Penalty Balance</dt>
                                    <dd class="text-sm font-bold text-penalty">K {{ number_format($loan->penalty_balance, 2) }}</dd>
                                </div>
                                <div class="flex justify-between items-center">
                                    <dt class="text-sm text-gray-600">Processing Fees</dt>
                                    <dd class="text-sm font-bold text-processing">K {{ number_format($loan->processing_fee_balance, 2) }}</dd>
                                </div>
                                <div class="pt-3 border-t border-gray-200">
                                    <div class="flex justify-between items-center">
                                        <dt class="text-sm font-medium text-gray-800">Total Outstanding</dt>
                                        <dd class="text-lg font-bold text-total">K {{ number_format($loan->totalbalance, 2) }}</dd>
                                    </div>
                                </div>
                                <div class="pt-3 border-t border-gray-200">
                                    <div class="flex justify-between items-center">
                                        <dt class="text-sm font-medium text-gray-800">Total Paid</dt>
                                        <dd class="text-lg font-bold text-green-600">K {{ number_format($loan->totalpaid, 2) }}</dd>
                                    </div>
                                    <div class="text-xs text-gray-500 mt-1">
                                        <div class="grid grid-cols-2 gap-2">
                                            <span>Principal: K {{ number_format($loan->repayments->sum('rprincipal'), 2) }}</span>
                                            <span>Interest: K {{ number_format($loan->repayments->sum('rinterest'), 2) }}</span>
                                            <span>Penalties: K {{ number_format($loan->repayments->sum('penalties'), 2) }}</span>
                                            <span>Processing: K {{ number_format($loan->repayments->sum('processing_fees_amount_paid'), 2) }}</span>
                                        </div>
                                    </div>
                                </div>
                            </dl>
                            
                            {{-- Repayment Progress --}}
                            @if($loan->amount > 0)
                                <div class="mt-6">
                                    <div class="flex justify-between items-center mb-2">
                                        <span class="text-sm font-medium text-gray-700">Repayment Progress</span>
                                        <span class="text-sm font-medium text-gray-700">
                                            {{ number_format(($loan->totalpaid / $loan->amount) * 100, 1) }}%
                                        </span>
                                    </div>
                                    <div class="w-full bg-gray-200 rounded-full h-2">
                                        @php
                                            $repaymentPercentage = min(($loan->totalpaid / $loan->amount) * 100, 100);
                                        @endphp
                                        <div class="bg-green-600 h-2 rounded-full" style="width: {{ $repaymentPercentage }}%"></div>
                                    </div>
                                    <div class="flex justify-between text-xs text-gray-500 mt-1">
                                        <span>K 0</span>
                                        <span>K {{ number_format($loan->amount, 2) }}</span>
                                    </div>
                                </div>
                            @endif
                        </div>
                    </div>
                    
                    {{-- Loan Timeline Card --}}
                    <div class="bg-white rounded-lg shadow-sm border border-gray-200 overflow-hidden">
                        <div class="border-b border-gray-200 px-6 py-4">
                            <h2 class="text-xl font-semibold text-gray-800">Loan Timeline</h2>
                        </div>
                        <div class="p-6">
                            <div class="space-y-4">
                                <div class="flex items-start">
                                    <div class="flex-shrink-0">
                                        <div class="w-8 h-8 bg-green-100 rounded-full flex items-center justify-center">
                                            <i class="fas fa-check text-green-600 text-sm"></i>
                                        </div>
                                    </div>
                                    <div class="ml-3">
                                        <p class="text-sm font-medium text-gray-900">Disbursed</p>
                                        <p class="text-sm text-gray-500">
                                            {{ $loan->disburseddate ? \Carbon\Carbon::parse($loan->disburseddate)->format('M d, Y') : 'N/A' }}
                                        </p>
                                    </div>
                                </div>
                                
                                <div class="flex items-start">
                                    <div class="flex-shrink-0">
                                        <div class="w-8 h-8 bg-blue-100 rounded-full flex items-center justify-center">
                                            <i class="fas fa-calendar text-blue-600 text-sm"></i>
                                        </div>
                                    </div>
                                    <div class="ml-3">
                                        <p class="text-sm font-medium text-gray-900">First Payment Due</p>
                                        <p class="text-sm text-gray-500">
                                            {{ $loan->firstpaymentdate ? \Carbon\Carbon::parse($loan->firstpaymentdate)->format('M d, Y') : 'N/A' }}
                                        </p>
                                    </div>
                                </div>
                                
                                <div class="flex items-start">
                                    <div class="flex-shrink-0">
                                        <div class="w-8 h-8 bg-purple-100 rounded-full flex items-center justify-center">
                                            <i class="fas fa-flag-checkered text-purple-600 text-sm"></i>
                                        </div>
                                    </div>
                                    <div class="ml-3">
                                        <p class="text-sm font-medium text-gray-900">Maturity Date</p>
                                        <p class="text-sm text-gray-500">
                                            {{ $loan->maturitydate ? \Carbon\Carbon::parse($loan->maturitydate)->format('M d, Y') : 'N/A' }}
                                        </p>
                                    </div>
                                </div>
                                
                                @if($loan->lastpaymentdate)
                                    <div class="flex items-start">
                                        <div class="flex-shrink-0">
                                            <div class="w-8 h-8 bg-green-100 rounded-full flex items-center justify-center">
                                                <i class="fas fa-dollar-sign text-green-600 text-sm"></i>
                                            </div>
                                        </div>
                                        <div class="ml-3">
                                            <p class="text-sm font-medium text-gray-900">Last Payment</p>
                                            <p class="text-sm text-gray-500">
                                                {{ \Carbon\Carbon::parse($loan->lastpaymentdate)->format('M d, Y') }}
                                                <span class="block text-xs">K {{ number_format($loan->lastpaymentamount, 2) }}</span>
                                            </p>
                                        </div>
                                    </div>
                                @endif
                                
                                @if($loan->closeddate)
                                    <div class="flex items-start">
                                        <div class="flex-shrink-0">
                                            <div class="w-8 h-8 bg-gray-100 rounded-full flex items-center justify-center">
                                                <i class="fas fa-lock text-gray-600 text-sm"></i>
                                            </div>
                                        </div>
                                        <div class="ml-3">
                                            <p class="text-sm font-medium text-gray-900">Loan Closed</p>
                                            <p class="text-sm text-gray-500">
                                                {{ \Carbon\Carbon::parse($loan->closeddate)->format('M d, Y') }}
                                            </p>
                                        </div>
                                    </div>
                                @endif
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </main>
    </div>

    <script>
        // Toggle payment schedule section
        function toggleSchedule() {
            const scheduleSection = document.getElementById('scheduleSection');
            const toggleBtn = document.getElementById('toggleScheduleBtn');
            
            if (scheduleSection.classList.contains('hidden')) {
                scheduleSection.classList.remove('hidden');
                toggleBtn.innerHTML = '<i class="fas fa-chevron-up mr-2"></i> Hide Schedule';
            } else {
                scheduleSection.classList.add('hidden');
                toggleBtn.innerHTML = '<i class="fas fa-chevron-down mr-2"></i> View Schedule';
            }
        }
        
        // Print loan details
        function printLoanDetails() {
            window.print();
        }
        
        // Confirm loan deletion
        function confirmDelete() {
            if (confirm('Are you sure you want to delete this loan? This action cannot be undone.')) {
                document.getElementById('deleteLoanForm').submit();
            }
        }
    </script>
</body>
</html>