<?php

use Illuminate\Support\Facades\Route;
use App\Http\Controllers\Api\CustomerAuthController;
use App\Http\Controllers\CustomerDashboardController;
use App\Http\Controllers\Api\PaymentApiController;
use App\Http\Controllers\Api\WebhookController;
use App\Http\Controllers\Api\LoanRepaymentController;
use App\Http\Controllers\Api\TransactionHistoryController;
use App\Http\Controllers\Api\LoanController;

// Test route
Route::get('/test', function () {
    return response()->json([
        'success' => true,
        'message' => 'API routes are working correctly.',
        'timestamp' => now()->toDateTimeString(),
    ]);
});

Route::get('/health', function () {
    return response()->json(['status' => 'ok', 'version' => '1.0.0']);
});

// ==============================================================================
// LOAN REPAYMENT API ROUTES
// ==============================================================================
Route::prefix('loans')->group(function () {
    Route::get('{loan}/repayments', [LoanRepaymentController::class, 'index']);
    Route::post('{loan}/storerepayment', [LoanRepaymentController::class, 'storefromweborapi']);
    Route::get('{loan}/repayments/{repayment}', [LoanRepaymentController::class, 'show']);
    Route::post('{loan}/calculate-allocation', [LoanRepaymentController::class, 'calculateAllocation']);
    Route::post('{loan}/repayments/{repayment}/reverse', [LoanRepaymentController::class, 'reverse']);
    Route::get('{loan}/statement', [StatementController::class, 'getLoanStatement']);
    Route::get('{loan}/payment-schedule', [StatementController::class, 'getPaymentSchedule']);
    Route::get('{loan}/transactions', [StatementController::class, 'getLoanTransactions']);
    Route::get('{loan}/balance-summary', [StatementController::class, 'getLoanBalanceSummary']);
    Route::get('customer/loans', [LoanController::class, 'getCustomerLoans']);
});

// ==============================================================================
// PUBLIC PAYMENT API ROUTES
// ==============================================================================
Route::prefix('payments')->group(function () {
    Route::post('/collect', [PaymentApiController::class, 'collect']);
    Route::post('/disburse', [PaymentApiController::class, 'disburse']);
    Route::get('/status/{referenceId}', [PaymentApiController::class, 'checkPaymentStatusPublic']);
    Route::get('/history/collections', [PaymentApiController::class, 'getCollectionHistory']);
    Route::get('/history/disbursements', [PaymentApiController::class, 'getDisbursementHistory']);
    Route::get('/stats', [PaymentApiController::class, 'getPaymentStats']);
    Route::get('/refresh/{referenceId}', [PaymentApiController::class, 'refreshPaymentStatus']);
    Route::get('/loan-last-payment', [PaymentApiController::class, 'checkLoanLastPayment']);
    Route::get('/loan-number-payment', [PaymentApiController::class, 'checkPaymentByLoanNumber']);
});

// ==============================================================================
// CUSTOMER API ROUTES
// ==============================================================================
Route::prefix('customer')->group(function () {

    // 1. AUTHENTICATION & REGISTRATION
    Route::prefix('register')->group(function () {
        Route::post('/step1', [CustomerAuthController::class, 'apiStep1']);
        Route::post('/step2', [CustomerAuthController::class, 'apiStep2']);
        Route::post('/step3', [CustomerAuthController::class, 'apiStep3']);
        Route::post('/next-of-kin', [CustomerAuthController::class, 'apiNextOfKin']);
        Route::post('/employment', [CustomerAuthController::class, 'apiEmployment']);
        Route::post('/step4', [CustomerAuthController::class, 'apiStep4']);
        Route::get('/progress/{token}', [CustomerAuthController::class, 'apiGetProgress']);
        Route::get('/resume/{token}', [CustomerAuthController::class, 'apiResumeRegistration']);
        Route::delete('/cancel/{token}', [CustomerAuthController::class, 'apiCancelRegistration']);
    });
    
    // NEW: ACCOUNT SYNCHRONIZATION ROUTES (Existing Customers)
    // --------------------------------------------------------------------------
    Route::prefix('sync')->group(function () {
        // Step 1: Find customer by phone and send OTP
        Route::post('/lookup', [CustomerAuthController::class, 'apiSyncLookup']);
        // Step 2: Verify OTP and return full profile for local DB storage
        Route::post('/verify', [CustomerAuthController::class, 'apiSyncVerify']);
    });
    
    // Publicly accessible profile update endpoints (Lookup via customer_id)
    Route::prefix('profile')->group(function () {
    
        // Updates text fields (Personal, NOK, Employment)
        Route::post('update', [CustomerAuthController::class, 'updateProfile']);
    
        // Handles Multipart/form-data for image uploads
        Route::post('photo', [CustomerAuthController::class, 'uploadProfilePhoto']);
     
    });

    Route::post('/login', [CustomerAuthController::class, 'apiLogin']);
    Route::post('/verify-otp', [CustomerAuthController::class, 'apiVerifyOtp']);
    Route::post('/resend-otp', [CustomerAuthController::class, 'apiResendOtp']);
    Route::post('/forgot-password', [CustomerAuthController::class, 'apiForgotPassword']);
    Route::post('/reset-password', [CustomerAuthController::class, 'apiResetPassword']);

    // Stage 2: Available products
    Route::get('/available-products', [CustomerDashboardController::class, 'getAvailableProducts']);

    // Stage 3 & 4: Specific details/requirements
    Route::get('/product-details/{id}', [CustomerDashboardController::class, 'getProductDetails']);

    // Stage 5: Eligibility
    Route::post('/check-eligibility', [CustomerDashboardController::class, 'checkEligibility']);

    // --------------------------------------------------------------------------
    // NEW: LOAN APPLICATION OTP SIGNATURE ROUTES
    // --------------------------------------------------------------------------
    // Request OTP to sign the application (Stores data in Cache temporarily)
    Route::post('/loan-application/request-otp', [CustomerDashboardController::class, 'requestLoanOtp']);
    
    // Verify OTP and officially Create/Submit the application
    Route::post('/loan-application/verify-otp', [CustomerDashboardController::class, 'verifyLoanOtp']);

    // 2. DASHBOARD & DATA
    Route::get('/dashboard', [CustomerDashboardController::class, 'getDashboardData']);
    Route::get('/transactions', [TransactionHistoryController::class, 'getHistory']);
    Route::get('/application-stats', [CustomerDashboardController::class, 'getApplicationStats']);
    Route::get('/loan-stats', [CustomerDashboardController::class, 'getLoanStats']);
    Route::get('/payments/sync-orphans', [CustomerDashboardController::class, 'syncOrphanedDisbursements']);
    
    // Applications
    Route::get('/applications', [CustomerDashboardController::class, 'getApplications']);
    Route::get('/applications/{id}', [CustomerDashboardController::class, 'getApplicationDetails']);
    Route::post('/applications/submit', [CustomerDashboardController::class, 'submitLoanApplication']);
    Route::post('/small-loan/submit', [CustomerDashboardController::class, 'smallloanapplication']);
    Route::post('/applications/{id}/withdraw', [CustomerDashboardController::class, 'withdrawApplication']);
    
    Route::get('/applications/status/{appNumber}', [CustomerDashboardController::class, 'getStatusByNumber']);
    
    // Loans
    Route::get('/loans', [LoanController::class, 'getCustomerLoans']);
    Route::get('/loans/{id}', [CustomerDashboardController::class, 'getLoanDetails']);
    
    // Payments
    Route::get('/payments/upcoming', [CustomerDashboardController::class, 'getUpcomingPaymentsApi']);
    Route::post('/payments/check-status', [CustomerDashboardController::class, 'checkPaymentStatus']);
    
    // Credit & Profile
    Route::get('/credit-score', [CustomerDashboardController::class, 'getCreditScore']);
    Route::get('/credit-score/history', [CustomerDashboardController::class, 'getCreditScoreHistory']);
    Route::get('/profile', [CustomerDashboardController::class, 'getProfile']);
    Route::get('/transactions/history', [StatementController::class, 'getCustomerTransactionHistory']);
    Route::get('/statement', [StatementController::class, 'getCustomerStatement']);

    // 3. PROTECTED ROUTES (Requires Bearer Token)
    Route::middleware(['auth:sanctum'])->group(function () {
        Route::post('/logout', [CustomerAuthController::class, 'apiLogout']);
        Route::get('/check-auth', [CustomerAuthController::class, 'apiCheckAuth']);
        Route::put('/profile', [CustomerAuthController::class, 'apiUpdateProfile']);
        Route::post('/profile/photo', [CustomerAuthController::class, 'uploadProfilePhoto']);
        Route::post('/change-password', [CustomerAuthController::class, 'apiChangePassword']);
        Route::post('/verify-phone', [CustomerAuthController::class, 'apiVerifyPhone']);
    });
});

// ==============================================================================
// PAYMENT WEBHOOK ROUTES
// ==============================================================================
Route::prefix('webhooks')->group(function () {
    Route::post('/lipila/collection', [WebhookController::class, 'handleCollectionCallback']);
    Route::post('/lipila/disbursement', [WebhookController::class, 'handleDisbursementCallback']);
    Route::get('/test', [WebhookController::class, 'testWebhook']);
    Route::get('/logs', [WebhookController::class, 'viewLogs']);
});