<?php
    // Define page variables
    $pageTitle = 'Closed Loans';
    $pageSubtitle = 'View all closed loans with settlement details';

    // Calculate dynamic stats for the top cards
    $totalPrincipal = $loans->sum('amount');
    $totalBalance = $loans->sum('principalbalance');
    $totalInterest = $loans->sum('interestbalance');
    $totalProcessing = $loans->sum('processing_fee_balance');
    $totalRepaid = $totalPrincipal - $totalBalance;
    $totalOutstanding = $totalBalance + $totalInterest + $totalProcessing;
    
    // Closed loan specific stats
    $totalWriteOffs = $loans->where('closure_reason', 'write_off_approved')->count();
    $totalSettled = $loans->where('closure_reason', 'fully_settled')->count();
    $totalForgiven = $loans->where('closure_reason', 'loan_forgiveness')->count();
    $totalWriteOffAmount = $loans->sum('closure_settlement_amount') ?? 0;
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><?php echo e($pageTitle); ?> | IMMIA FINANCE</title>
    
    <script src="https://cdn.tailwindcss.com"></script>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    
    <style>
        /* --- ZEDBANKOS THEME COLORS --- */
        :root {
            --zedbank-blue: #0077C5; /* ThemeBlue */
            --zedbank-blue-dark: #005A94; /* Darker shade for hover/active background */
        }

        /* Custom Tailwind-like classes */
        .bg-zedbank-blue { background-color: var(--zedbank-blue); }
        .bg-zedbank-blue-dark { background-color: var(--zedbank-blue-dark); }
        .text-zedbank-blue { color: var(--zedbank-blue); }
        .hover\:bg-zedbank-blue-dark:hover { background-color: var(--zedbank-blue-dark); }

        /* Sidebar/Main Content Layout */
        .lg\:ml-64 { margin-left: 16rem; }
        @media (max-width: 1024px) { 
            .lg\:ml-64 { margin-left: 0 !important; }
        }

        /* Local Overrides */
        .focus\:ring-primary:focus { --tw-ring-color: var(--zedbank-blue); }
        .focus\:border-primary:focus { border-color: var(--zedbank-blue); }
        .bg-primary { background-color: var(--zedbank-blue); }
        
        /* Custom colors for closed loans */
        .text-closed { color: #6B7280; }
        .bg-closed-light { background-color: #F9FAFB; }
        .text-settled { color: #059669; }
        .bg-settled-light { background-color: #D1FAE5; }
        .text-writeoff { color: #DC2626; }
        .bg-writeoff-light { background-color: #FEE2E2; }
        .text-forgiven { color: #7C3AED; }
        .bg-forgiven-light { background-color: #EDE9FE; }
    </style>
</head>
<body class="bg-gray-50 min-h-screen">
    
    
    <?php echo $__env->make('accounting.partials.sidebar', array_diff_key(get_defined_vars(), ['__data' => 1, '__path' => 1]))->render(); ?>

    <div class="lg:ml-64 min-h-screen">
        
        
        <?php echo $__env->make('accounting.partials.header', [
            'title' => $pageTitle,
            'subtitle' => $pageSubtitle
        ], array_diff_key(get_defined_vars(), ['__data' => 1, '__path' => 1]))->render(); ?>

        <main class="p-6">
            <div class="mb-8">
                <h1 class="text-3xl font-bold text-gray-800 mb-2">Closed Loans</h1>
                <p class="text-gray-600">Overview of all loans that have been closed or settled.</p>
            </div>
            
            <?php if(session('success')): ?>
                <div class="mb-6 bg-green-50 border border-green-200 text-green-700 px-4 py-3 rounded-lg">
                    <?php echo e(session('success')); ?>

                </div>
            <?php endif; ?>
            
            <?php if(session('error')): ?>
                <div class="mb-6 bg-red-50 border border-red-200 text-red-700 px-4 py-3 rounded-lg">
                    <?php echo e(session('error')); ?>

                </div>
            <?php endif; ?>

            
            <div class="grid grid-cols-1 md:grid-cols-6 gap-4 mb-6">
                <div class="bg-white rounded-lg p-4 shadow-sm border">
                    <div class="flex items-center justify-between">
                        <div>
                            <p class="text-sm text-gray-600">Closed Loans</p>
                            <p class="text-2xl font-bold text-zedbank-blue"><?php echo e($loans->total()); ?></p>
                        </div>
                        <div class="w-10 h-10 bg-blue-100 rounded-lg flex items-center justify-center">
                            <i class="fas fa-archive text-zedbank-blue"></i>
                        </div>
                    </div>
                </div>
                
                <div class="bg-white rounded-lg p-4 shadow-sm border">
                    <div class="flex items-center justify-between">
                        <div>
                            <p class="text-sm text-gray-600">Fully Settled</p>
                            <p class="text-lg font-bold text-settled"><?php echo e($totalSettled); ?></p>
                            <p class="text-xs text-gray-500 mt-1">Normal closure</p>
                        </div>
                        <div class="w-10 h-10 bg-settled-light rounded-lg flex items-center justify-center">
                            <i class="fas fa-check-circle text-settled"></i>
                        </div>
                    </div>
                </div>
                
                <div class="bg-white rounded-lg p-4 shadow-sm border">
                    <div class="flex items-center justify-between">
                        <div>
                            <p class="text-sm text-gray-600">Write-offs</p>
                            <p class="text-lg font-bold text-writeoff"><?php echo e($totalWriteOffs); ?></p>
                            <?php if($totalWriteOffAmount > 0): ?>
                                <p class="text-xs text-gray-500 mt-1"><?php echo e(number_format($totalWriteOffAmount, 2)); ?></p>
                            <?php endif; ?>
                        </div>
                        <div class="w-10 h-10 bg-writeoff-light rounded-lg flex items-center justify-center">
                            <i class="fas fa-file-invoice text-writeoff"></i>
                        </div>
                    </div>
                </div>
                
                <div class="bg-white rounded-lg p-4 shadow-sm border">
                    <div class="flex items-center justify-between">
                        <div>
                            <p class="text-sm text-gray-600">Loan Forgiveness</p>
                            <p class="text-lg font-bold text-forgiven"><?php echo e($totalForgiven); ?></p>
                            <p class="text-xs text-gray-500 mt-1">Special cases</p>
                        </div>
                        <div class="w-10 h-10 bg-forgiven-light rounded-lg flex items-center justify-center">
                            <i class="fas fa-hand-holding-heart text-forgiven"></i>
                        </div>
                    </div>
                </div>
                
                <div class="bg-white rounded-lg p-4 shadow-sm border">
                    <div class="flex items-center justify-between">
                        <div>
                            <p class="text-sm text-gray-600">Total Principal</p>
                            <p class="text-lg font-bold text-gray-800"><?php echo e(number_format($totalPrincipal, 2)); ?></p>
                        </div>
                        <div class="w-10 h-10 bg-gray-100 rounded-lg flex items-center justify-center">
                            <i class="fas fa-home text-gray-600"></i>
                        </div>
                    </div>
                </div>
                
                <div class="bg-white rounded-lg p-4 shadow-sm border">
                    <div class="flex items-center justify-between">
                        <div>
                            <p class="text-sm text-gray-600">Total Repaid</p>
                            <p class="text-lg font-bold text-green-600"><?php echo e(number_format($totalRepaid, 2)); ?></p>
                            <?php if($totalPrincipal > 0): ?>
                                <p class="text-xs text-gray-500 mt-1">
                                    <?php echo e(number_format(($totalRepaid / $totalPrincipal) * 100, 1)); ?>%
                                </p>
                            <?php endif; ?>
                        </div>
                        <div class="w-10 h-10 bg-green-100 rounded-lg flex items-center justify-center">
                            <i class="fas fa-calculator text-green-600"></i>
                        </div>
                    </div>
                </div>
            </div>

            
            <div class="bg-white rounded-lg shadow-sm border border-gray-200 p-4 mb-6">
                <div class="flex flex-col md:flex-row justify-between items-start md:items-center gap-4">
                    <div class="w-full md:w-auto">
                        <div class="relative">
                            <div class="absolute inset-y-0 left-0 pl-3 flex items-center pointer-events-none">
                                <i class="fas fa-search text-gray-400"></i>
                            </div>
                            <input 
                                type="text" 
                                id="searchInput" 
                                class="block w-full md:w-80 pl-10 pr-4 py-2 border border-gray-300 rounded-lg focus:ring-primary focus:border-primary" 
                                placeholder="Search by loan number, customer, reason..."
                            >
                        </div>
                    </div>
                    <div class="flex flex-wrap gap-2 w-full md:w-auto">
                        <a href="<?php echo e(route('admin.loans.active')); ?>" 
                           class="px-4 py-2 bg-green-600 text-white rounded-lg hover:bg-green-700 transition flex items-center gap-2">
                            <i class="fas fa-chart-line"></i>
                            <span>Active Loans</span>
                        </a>
                        <a href="<?php echo e(route('admin.loans.non-performing')); ?>" 
                           class="px-4 py-2 bg-red-600 text-white rounded-lg hover:bg-red-700 transition flex items-center gap-2">
                            <i class="fas fa-exclamation-triangle"></i>
                            <span>NPL Loans</span>
                        </a>
                        <button class="px-4 py-2 bg-white border border-gray-300 rounded-lg hover:bg-gray-50 transition flex items-center gap-2" id="filterBtn">
                            <i class="fas fa-filter"></i>
                            <span>Filter by Reason</span>
                        </button>
                        <button class="px-4 py-2 bg-white border border-gray-300 rounded-lg hover:bg-gray-50 transition flex items-center gap-2" id="exportBtn">
                            <i class="fas fa-download"></i>
                            <span>Export</span>
                        </button>
                    </div>
                </div>
            </div>

            
            <div class="overflow-x-auto bg-white shadow-sm rounded-lg border border-gray-200">
                <table class="min-w-full text-sm" id="loansTable">
                    <thead class="bg-gray-50 border-b border-gray-200">
                        <tr>
                            <th class="px-4 py-3 text-left text-gray-600 font-medium">Loan No</th>
                            <th class="px-4 py-3 text-left text-gray-600 font-medium">Customer</th>
                            <th class="px-4 py-3 text-left text-gray-600 font-medium">Product</th>
                            <th class="px-4 py-3 text-left text-gray-600 font-medium">Disbursed</th>
                            <th class="px-4 py-3 text-left text-gray-600 font-medium">Closed Date</th>
                            <th class="px-4 py-3 text-left text-gray-600 font-medium">Principal</th>
                            <th class="px-4 py-3 text-left text-gray-600 font-medium">Settlement Amount</th>
                            <th class="px-4 py-3 text-left text-gray-600 font-medium">Reason</th>
                            <th class="px-4 py-3 text-left text-gray-600 font-medium">Term</th>
                            <th class="px-4 py-3 text-left text-gray-600 font-medium">Status</th>
                            <th class="px-4 py-3 text-left text-gray-600 font-medium">Closed By</th>
                            <th class="px-4 py-3 text-left text-gray-600 font-medium">Actions</th>
                        </tr>
                    </thead>

                    <tbody>
                        <?php $__empty_1 = true; $__currentLoopData = $loans; $__env->addLoop($__currentLoopData); foreach($__currentLoopData as $loan): $__env->incrementLoopIndices(); $loop = $__env->getLastLoop(); $__empty_1 = false; ?>
                            <?php
                                // Map closure reason to display text and colors
                                $closureReasons = [
                                    'fully_settled' => ['text' => 'Fully Settled', 'color' => 'text-settled', 'bg' => 'bg-settled-light', 'icon' => 'fa-check-circle'],
                                    'loan_forgiveness' => ['text' => 'Loan Forgiveness', 'color' => 'text-forgiven', 'bg' => 'bg-forgiven-light', 'icon' => 'fa-hand-holding-heart'],
                                    'settlement_agreement' => ['text' => 'Settlement Agreement', 'color' => 'text-blue-600', 'bg' => 'bg-blue-100', 'icon' => 'fa-handshake'],
                                    'write_off_approved' => ['text' => 'Write-off', 'color' => 'text-writeoff', 'bg' => 'bg-writeoff-light', 'icon' => 'fa-file-invoice'],
                                    'customer_bankruptcy' => ['text' => 'Customer Bankruptcy', 'color' => 'text-orange-600', 'bg' => 'bg-orange-100', 'icon' => 'fa-user-times'],
                                    'deceased_customer' => ['text' => 'Deceased Customer', 'color' => 'text-gray-600', 'bg' => 'bg-gray-100', 'icon' => 'fa-cross'],
                                    'system_error' => ['text' => 'System Error', 'color' => 'text-red-600', 'bg' => 'bg-red-100', 'icon' => 'fa-bug'],
                                    'other' => ['text' => 'Other', 'color' => 'text-gray-600', 'bg' => 'bg-gray-100', 'icon' => 'fa-ellipsis-h'],
                                ];
                                
                                $reason = $loan->closure_reason ?? 'other';
                                $reasonInfo = $closureReasons[$reason] ?? $closureReasons['other'];
                                
                                // Check if loan has write-off amount
                                $hasWriteOff = $loan->closure_settlement_amount > 0 && $loan->closure_settlement_amount < $loan->amount;
                                $writeOffAmount = $hasWriteOff ? ($loan->amount - $loan->closure_settlement_amount) : 0;
                                
                                // Get closed by user info
                                $closedBy = $loan->closedBy ?? null;
                            ?>
                            <tr class="border-b hover:bg-gray-50 transition">
                                <td class="px-4 py-3 text-gray-700 font-medium">
                                    <?php echo e($loan->loannumber); ?>

                                </td>
                                <td class="px-4 py-3 text-gray-700">
                                    <?php if($loan->customer): ?>
                                        <p class="font-semibold"><?php echo e($loan->customer->first_name); ?> <?php echo e($loan->customer->surname); ?></p>
                                        <p class="text-xs text-gray-500"><?php echo e($loan->customer->customer_number); ?></p>
                                    <?php else: ?>
                                        <span class="text-gray-500">—</span>
                                    <?php endif; ?>
                                </td>
                                <td class="px-4 py-3 text-gray-700">
                                    <?php echo e($loan->loanType->product ?? 'N/A'); ?>

                                </td>
                                <td class="px-4 py-3 text-gray-700">
                                    <?php echo e($loan->disburseddate ? \Carbon\Carbon::parse($loan->disburseddate)->format('M d, Y') : 'N/A'); ?>

                                </td>
                                <td class="px-4 py-3 text-gray-700 font-medium">
                                    <?php echo e($loan->closeddate ? \Carbon\Carbon::parse($loan->closeddate)->format('M d, Y') : 'N/A'); ?>

                                </td>
                                <td class="px-4 py-3 text-gray-800 font-bold">
                                    <div class="flex flex-col">
                                        <span><?php echo e(number_format($loan->amount, 2)); ?></span>
                                        <?php if($hasWriteOff): ?>
                                            <span class="text-xs text-red-600">
                                                <i class="fas fa-exclamation-circle mr-1"></i>
                                                Write-off: <?php echo e(number_format($writeOffAmount, 2)); ?>

                                            </span>
                                        <?php endif; ?>
                                    </div>
                                </td>
                                <td class="px-4 py-3">
                                    <div class="flex flex-col">
                                        <span class="font-bold <?php echo e($hasWriteOff ? 'text-red-600' : 'text-green-600'); ?>">
                                            <?php echo e($loan->closure_settlement_amount ? number_format($loan->closure_settlement_amount, 2) : number_format($loan->amount, 2)); ?>

                                        </span>
                                        <?php if($loan->closure_settlement_amount && $loan->closure_settlement_amount != $loan->amount): ?>
                                            <span class="text-xs text-gray-500">
                                                <?php echo e(number_format(($loan->closure_settlement_amount / $loan->amount) * 100, 1)); ?>% of original
                                            </span>
                                        <?php endif; ?>
                                    </div>
                                </td>
                                <td class="px-4 py-3">
                                    <span class="inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium <?php echo e($reasonInfo['bg']); ?> <?php echo e($reasonInfo['color']); ?>">
                                        <i class="fas <?php echo e($reasonInfo['icon']); ?> mr-1"></i>
                                        <?php echo e($reasonInfo['text']); ?>

                                    </span>
                                    <?php if($loan->closure_notes): ?>
                                        <button class="ml-1 text-gray-400 hover:text-gray-600" 
                                                onclick="showNotes('<?php echo e(addslashes($loan->closure_notes)); ?>', '<?php echo e($reasonInfo['text']); ?>')"
                                                title="View closure notes">
                                            <i class="fas fa-sticky-note"></i>
                                        </button>
                                    <?php endif; ?>
                                </td>
                                <td class="px-4 py-3 text-gray-700">
                                    <div class="flex flex-col">
                                        <span class="font-medium"><?php echo e($loan->loanterm); ?> <?php echo e($loan->paymentfrequency === 'monthly' ? 'Months' : ucfirst($loan->paymentfrequency)); ?></span>
                                        <span class="text-xs text-gray-500">
                                            <?php echo e(number_format($loan->interestrate, 2)); ?>%
                                        </span>
                                    </div>
                                </td>
                                <td class="px-4 py-3">
                                    <span class="inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium bg-gray-100 text-gray-800">
                                        <i class="fas fa-lock mr-1"></i>
                                        Closed
                                    </span>
                                </td>
                                <td class="px-4 py-3 text-gray-700">
                                    <?php if($closedBy): ?>
                                        <div class="flex flex-col">
                                            <span class="font-medium"><?php echo e($closedBy->name); ?></span>
                                            <span class="text-xs text-gray-500">
                                                <?php echo e($loan->closeddate ? \Carbon\Carbon::parse($loan->closeddate)->format('M d, Y') : ''); ?>

                                            </span>
                                        </div>
                                    <?php else: ?>
                                        <span class="text-gray-500">—</span>
                                    <?php endif; ?>
                                </td>
                                <td class="px-4 py-3">
                                    <div class="flex space-x-2">
                                        <a href="<?php echo e(route('admin.loans.show', $loan->loanid)); ?>" 
                                           class="text-zedbank-blue hover:text-zedbank-blue-dark p-1 rounded transition" 
                                           title="View Details">
                                            <i class="fas fa-eye"></i>
                                        </a>
                                        
                                        <?php if(auth()->user()->hasPermission('loans.reverse')): ?>
                                            <form method="POST" action="<?php echo e(route('admin.loans.reverse', $loan->loanid)); ?>" 
                                                  class="inline" 
                                                  onsubmit="return confirmReverse('<?php echo e($loan->loannumber); ?>')">
                                                <?php echo csrf_field(); ?>
                                                <button type="submit" 
                                                        class="text-green-600 hover:text-green-800 p-1 rounded transition" 
                                                        title="Reopen Loan">
                                                    <i class="fas fa-undo-alt"></i>
                                                </button>
                                            </form>
                                        <?php endif; ?>
                                        
                                        <a href="<?php echo e(route('admin.loans.loan-statement', $loan->loanid)); ?>" 
                                           class="text-purple-600 hover:text-purple-800 p-1 rounded transition" 
                                           title="View Statement">
                                            <i class="fas fa-file-invoice-dollar"></i>
                                        </a>
                                    </div>
                                </td>
                            </tr>
                        <?php endforeach; $__env->popLoop(); $loop = $__env->getLastLoop(); if ($__empty_1): ?>
                            <tr>
                                <td colspan="12" class="px-4 py-8 text-center text-gray-500">
                                    <div class="flex flex-col items-center justify-center">
                                        <i class="fas fa-archive text-4xl text-gray-300 mb-4"></i>
                                        <p class="text-lg">No closed loans found</p>
                                        <p class="text-sm">Closed loans will appear here once they are marked as closed.</p>
                                        <a href="<?php echo e(route('admin.loans.active')); ?>" 
                                           class="mt-4 px-4 py-2 bg-zedbank-blue text-white rounded-lg hover:bg-zedbank-blue-dark transition">
                                            View Active Loans
                                        </a>
                                    </div>
                                </td>
                            </tr>
                        <?php endif; ?>
                    </tbody>
                </table>
            </div>

            
            <div class="mt-6 flex flex-col md:flex-row justify-between items-center gap-4">
                <div class="text-sm text-gray-600">
                    Showing <span class="font-medium"><?php echo e($loans->firstItem() ?? 0); ?></span> to 
                    <span class="font-medium"><?php echo e($loans->lastItem() ?? 0); ?></span> of 
                    <span class="font-medium"><?php echo e($loans->total()); ?></span> entries
                </div>
                <div class="flex space-x-2">
                    <?php if($loans->onFirstPage()): ?>
                        <button class="px-4 py-2 text-sm bg-white border border-gray-300 rounded-lg text-gray-400 cursor-not-allowed flex items-center gap-2" disabled>
                            <i class="fas fa-chevron-left"></i>
                            <span>Previous</span>
                        </button>
                    <?php else: ?>
                        <a href="<?php echo e($loans->previousPageUrl()); ?>" class="px-4 py-2 text-sm bg-white border border-gray-300 rounded-lg hover:bg-gray-50 transition flex items-center gap-2">
                            <i class="fas fa-chevron-left"></i>
                            <span>Previous</span>
                        </a>
                    <?php endif; ?>

                    <?php $__currentLoopData = $loans->getUrlRange(1, $loans->lastPage()); $__env->addLoop($__currentLoopData); foreach($__currentLoopData as $page => $url): $__env->incrementLoopIndices(); $loop = $__env->getLastLoop(); ?>
                        <?php if($page == $loans->currentPage()): ?>
                            <span class="px-4 py-2 text-sm bg-zedbank-blue text-white rounded-lg"><?php echo e($page); ?></span>
                        <?php else: ?>
                            <a href="<?php echo e($url); ?>" class="px-4 py-2 text-sm bg-white border border-gray-300 rounded-lg hover:bg-gray-50 transition"><?php echo e($page); ?></a>
                        <?php endif; ?>
                    <?php endforeach; $__env->popLoop(); $loop = $__env->getLastLoop(); ?>

                    <?php if($loans->hasMorePages()): ?>
                        <a href="<?php echo e($loans->nextPageUrl()); ?>" class="px-4 py-2 text-sm bg-white border border-gray-300 rounded-lg hover:bg-gray-50 transition flex items-center gap-2">
                            <span>Next</span>
                            <i class="fas fa-chevron-right"></i>
                        </a>
                    <?php else: ?>
                        <button class="px-4 py-2 text-sm bg-white border border-gray-300 rounded-lg text-gray-400 cursor-not-allowed flex items-center gap-2" disabled>
                            <span>Next</span>
                            <i class="fas fa-chevron-right"></i>
                        </button>
                    <?php endif; ?>
                </div>
            </div>
        </main>
    </div>

    
    <div id="notesModal" class="fixed inset-0 bg-gray-600 bg-opacity-50 overflow-y-auto h-full w-full z-50 hidden">
        <div class="relative top-20 mx-auto p-5 border w-96 shadow-lg rounded-lg bg-white">
            <div class="flex justify-between items-center mb-4">
                <h3 class="text-lg font-semibold text-gray-800" id="modalTitle">Closure Notes</h3>
                <button onclick="closeNotes()" class="text-gray-400 hover:text-gray-600">
                    <i class="fas fa-times"></i>
                </button>
            </div>
            <div class="mb-4">
                <p class="text-gray-700" id="modalNotes"></p>
            </div>
            <div class="flex justify-end">
                <button onclick="closeNotes()" class="px-4 py-2 bg-zedbank-blue text-white rounded-lg hover:bg-zedbank-blue-dark transition">
                    Close
                </button>
            </div>
        </div>
    </div>

    <script>
        document.addEventListener('DOMContentLoaded', function() {
            const searchInput = document.getElementById('searchInput');
            const loansTable = document.getElementById('loansTable');
            const loanRows = loansTable.querySelectorAll('tbody tr');
            const filterBtn = document.getElementById('filterBtn');
            const exportBtn = document.getElementById('exportBtn');

            // Search functionality
            searchInput?.addEventListener('input', function() {
                const searchTerm = this.value.toLowerCase();
                
                loanRows.forEach(row => {
                    const rowText = row.textContent.toLowerCase();
                    if (rowText.includes(searchTerm)) {
                        row.style.display = '';
                    } else {
                        row.style.display = 'none';
                    }
                });
            });

            // Filter button functionality
            filterBtn?.addEventListener('click', function() {
                const reasons = [
                    { value: 'all', text: 'All Reasons' },
                    { value: 'fully_settled', text: 'Fully Settled' },
                    { value: 'loan_forgiveness', text: 'Loan Forgiveness' },
                    { value: 'settlement_agreement', text: 'Settlement Agreement' },
                    { value: 'write_off_approved', text: 'Write-off' },
                    { value: 'customer_bankruptcy', text: 'Customer Bankruptcy' },
                    { value: 'deceased_customer', text: 'Deceased Customer' },
                    { value: 'system_error', text: 'System Error' },
                    { value: 'other', text: 'Other' }
                ];
                
                const filterText = prompt('Enter closure reason to filter by:\n\n' + 
                    reasons.map(r => `${r.value} - ${r.text}`).join('\n') + 
                    '\n\nOr type "all" to show all loans:');
                
                if (filterText) {
                    const filterTerm = filterText.toLowerCase().trim();
                    
                    loanRows.forEach(row => {
                        if (filterTerm === 'all') {
                            row.style.display = '';
                        } else {
                            const reasonCell = row.querySelector('td:nth-child(8)');
                            if (reasonCell) {
                                const reasonText = reasonCell.textContent.toLowerCase();
                                if (reasonText.includes(filterTerm)) {
                                    row.style.display = '';
                                } else {
                                    row.style.display = 'none';
                                }
                            }
                        }
                    });
                }
            });
            
            // Export button functionality
            exportBtn?.addEventListener('click', function() {
                alert('Exporting closed loans data...\n\nThis would generate a CSV/PDF report of all closed loans.');
            });
        });

        // Notes modal functions
        function showNotes(notes, reason) {
            document.getElementById('modalTitle').textContent = 'Closure Notes - ' + reason;
            document.getElementById('modalNotes').textContent = notes;
            document.getElementById('notesModal').classList.remove('hidden');
        }

        function closeNotes() {
            document.getElementById('notesModal').classList.add('hidden');
        }

        // Confirm loan reversal
        function confirmReverse(loanNumber) {
            return confirm(`Are you sure you want to reopen loan ${loanNumber}?\n\nThis will change the status back to "active" and allow further transactions.`);
        }

        // Close modal when clicking outside
        window.onclick = function(event) {
            const modal = document.getElementById('notesModal');
            if (event.target === modal) {
                closeNotes();
            }
        }
    </script>
</body>
</html><?php /**PATH /home/topacvbu/public_html/zedbankos.cloud/manageportal/resources/views/admin/loans/closed.blade.php ENDPATH**/ ?>